/**************************************************************************************

   Copyright (c) Hilscher GmbH. All Rights Reserved.

 **************************************************************************************

   Filename:
    $Workfile: Fieldbus_Functions.c $
   Last Modification:
    $Author: Robert $
    $Modtime:  $
    $Revision: 6597 $

   Targets:
     Win32/ANSI   : yes
     Win32/Unicode: no
     WinCE        : no

   Description:
     General example functions for file handling

   Changes:

     Version   Date        Author   Description
     ----------------------------------------------------------------------------------
     2         18.01.2013  RM       Added hardware function macros
     
     1         17.04.2012  RM       initial version

**************************************************************************************/
#include "cifXHWFunctions.h"
#include "OS_Dependent.h"
#include "OS_Includes.h"

/* Error & Type Def. */
#include "CIFXErrors.h"
#include "CIFXEndianess.h"

#include "rcX_Public.h"
#include "ProfibusAps_Public.h"

/* ============================================================================= */
/* Global infomation and definitions                                             */
/* ============================================================================= */
/* Global data */
extern uint32_t ulSourceID;
extern uint8_t  bHandleIndications;

/*****************************************************************************/
/*! Register Application
 * \return CIFX_NO_ERROR on success                                          */
/*****************************************************************************/
int32_t Fieldbus_RegisterApplication( PCHANNELINSTANCE ptChannel)
{
  /* Register Application to get Indications from the stack */
  int32_t                 lRet              = CIFX_NO_ERROR;
  CIFX_PACKET             tReceivePacket    = {0};
  RCX_REGISTER_APP_REQ_T  tRegisterAppReq   = {0};

  OS_Memset( &tRegisterAppReq,  0, sizeof (RCX_REGISTER_APP_REQ_T) );
  OS_Memset( &tReceivePacket,   0, sizeof(tReceivePacket));

  /* Create a register application request */
  tRegisterAppReq.tHead.ulDest          = HOST_TO_LE32(RCX_PACKET_DEST_DEFAULT_CHANNEL);
  tRegisterAppReq.tHead.ulSrc           = HOST_TO_LE32(ulSourceID);
  tRegisterAppReq.tHead.ulLen           = HOST_TO_LE32(0);
  tRegisterAppReq.tHead.ulCmd           = HOST_TO_LE32(RCX_REGISTER_APP_REQ);

  /* send configuration Req packet */
  lRet = DEV_TransferPacket(  ptChannel, (CIFX_PACKET*)&tRegisterAppReq, &tReceivePacket, 
                              sizeof(tReceivePacket), CIFX_TO_SEND_PACKET, NULL, NULL);

  /* check if we got an error */
  if (CIFX_NO_ERROR == lRet)
    lRet = tReceivePacket.tHeader.ulState;

  return lRet;
}

/*****************************************************************************/
/*! Handle Profibus Indication Packets
 * \param ptIndicationPacket  Pointer to the Indication packet              
 * \return !=0 a pcket was handled                                           */
/*****************************************************************************/
uint8_t ProfibusHandleIndication( DEVICEINSTANCE* pvDeviceInstance,
                                  CIFX_PACKET*    ptIndicationPacket,
                                  CIFX_PACKET*    ptResponsePacket)
{
  uint8_t      bState       = 0;
  CIFX_PACKET  tIndication  = {{0}};
  CIFX_PACKET* ptIndication = &tIndication;
  
  HWIF_READN( pvDeviceInstance, &tIndication, ptIndicationPacket, sizeof(CIFX_PACKET));

  switch (ptIndication->tHeader.ulCmd)
  {
    case PROFIBUS_APS_CHECK_CFG_IND:
      {
        /* We have to answer this indication */
        PROFIBUS_APS_PACKET_CHECK_CFG_IND_T* ptPBIndication = (PROFIBUS_APS_PACKET_CHECK_CFG_IND_T*)ptIndication;
        PROFIBUS_APS_PACKET_CHECK_CFG_RES_T  tResponse;
        PROFIBUS_APS_PACKET_CHECK_CFG_RES_T* ptResponse     = &tResponse; 

        /* Copy packet from the indication to the response */
        OS_Memcpy( ptResponse, ptPBIndication, sizeof( PROFIBUS_APS_PACKET_CHECK_CFG_RES_T));

        /* Insert a I/O length into the packet */
        ptResponse->tHead.ulCmd          = HOST_TO_LE32(PROFIBUS_APS_CHECK_CFG_RES);
        ptResponse->tHead.ulLen          = HOST_TO_LE32(sizeof(PROFIBUS_APS_CHECK_CFG_RES_T));
        ptResponse->tHead.ulSta          = HOST_TO_LE32(0);

        ptResponse->tData.fCfgOk = TRUE;
        
        /* Copy packet from the indication to the response */
        HWIF_WRITEN( pvDeviceInstance, ptResponsePacket, ptResponse, sizeof( PROFIBUS_APS_PACKET_CHECK_CFG_RES_T));

        bState = 1;  /* Check Config packet handled */
      }
      break;


    case PROFIBUS_APS_CHECK_USER_PRM_IND:
      {
        /* We have to answer this indication */
        PROFIBUS_APS_PACKET_CHECK_USER_PRM_IND_T* ptPBIndication = (PROFIBUS_APS_PACKET_CHECK_USER_PRM_IND_T*)ptIndication;
        PROFIBUS_APS_PACKET_CHECK_USER_PRM_RES_T  tResponse;
        PROFIBUS_APS_PACKET_CHECK_USER_PRM_RES_T* ptResponse     = &tResponse; 
        
        /* Copy packet from the indication to the response */
        OS_Memcpy( ptResponse, ptPBIndication, sizeof( PROFIBUS_APS_PACKET_CHECK_USER_PRM_RES_T));

        /* Insert a TRUE into the response to accept the packet */
        ptResponse->tHead.ulCmd = HOST_TO_LE32(PROFIBUS_APS_CHECK_USER_PRM_RES);
        ptResponse->tHead.ulLen = HOST_TO_LE32(sizeof(PROFIBUS_APS_CHECK_USER_PRM_RES_T));
        ptResponse->tHead.ulSta = HOST_TO_LE32(0);

        ptResponse->tData.fPrmOk = TRUE;
        
        /* Copy packet from the indication to the response */
        HWIF_WRITEN( pvDeviceInstance, ptResponsePacket, ptResponse, sizeof( PROFIBUS_APS_PACKET_CHECK_USER_PRM_RES_T));

        bState = 2;  /* Check parameter packet handled */
      }
      break;


    default:
      /* do nothing */
      break;
  }

  return bState;
}


/*****************************************************************************/
/*! Handle Fieldbus Indications
*
*   \return bState                                                           */
/*****************************************************************************/
uint8_t Fieldbus_HandleIndications( PCHANNELINSTANCE ptChannel)
{
  uint8_t bState = 0;

  /* Now we have to wait until the master has sent his indications */
  CIFX_PACKET* ptRecvPkt    = (CIFX_PACKET*)&ptChannel->tRecvMbx.ptRecvMailboxStart->abRecvMailbox[0];
  CIFX_PACKET* ptSentPacket = (CIFX_PACKET*)&ptChannel->tSendMbx.ptSendMailboxStart->abSendMailbox[0];

  /* Check if a packet is available and if the send mailbox is free */
  if ( (DEV_WaitForBitState(ptChannel, ptChannel->tRecvMbx.bRecvACKBitoffset, RCX_FLAGS_NOT_EQUAL, 0)) && 
       (DEV_WaitForBitState(ptChannel, ptChannel->tSendMbx.bSendCMDBitoffset, RCX_FLAGS_EQUAL, 0)) )
  {
    /* We have a packet in the receive mailbox and the send mailbox is empty */
    bState = ProfibusHandleIndication( ptChannel->pvDeviceInstance, ptRecvPkt, ptSentPacket);

    if (bState != 0)
      /* We have to send a packet */
      /* Signal sent a packet*/
      DEV_ToggleBit(ptChannel, ptChannel->tSendMbx.ulSendCMDBitmask);

    /* Signal read packet done */
    DEV_ToggleBit(ptChannel, ptChannel->tRecvMbx.ulRecvACKBitmask);
  }

  return bState;
}

/*****************************************************************************/
/*! Activate Fieldbus Indications PROFIBUS EXAMPLE
*
*   \return bState                                                           */
/*****************************************************************************/
int32_t Fieldbus_ActivateIndications( PCHANNELINSTANCE ptChannel,
                                      uint16_t         usProtocollClass,
                                      uint16_t         usCommunicationClass,
                                      uint8_t*         pbHandleIndications)
{
  int32_t lError = 0; 
  
  /*------------------------------------------------------------------------*/
  /* Special PROFIBUS handling                                              */
  /*------------------------------------------------------------------------*/
  /* PROFIBUS Indication Handling */
  /*------------------------------------------------------------------------*/
  /* COM Module Compatibility                                               */
  /* To drive a PROFIBUS COMX module with the same GSD file a               */
  /* PROFIBUS COM module we have to register as an application to get       */
  /* Indications from the master.                                           */
  /*------------------------------------------------------------------------*/
  if ( (RCX_PROT_CLASS_PROFIBUS_DP == usProtocollClass) &&
       (RCX_COMM_CLASS_SLAVE == usCommunicationClass)    )
  {
    uint8_t bIndicationState = 0;

    /* We have to register the application to get indications from the fieldbus */
    lError = Fieldbus_RegisterApplication( ptChannel);
    if( (CIFX_NO_ERROR == lError) ||
        (0xC0000201    == lError)  )     /* Application already registered */
    {
      /* Successfully registered for indications */
      *pbHandleIndications = TRUE;
      lError               = 0;

      /* Now we have to wait until we got the necessary indications before we can start */
      /* with I/O data handling */
      do
      {
        uint8_t bState = Fieldbus_HandleIndications( ptChannel);

        bIndicationState |= bState;

        OS_Sleep(100);

      } while (bIndicationState != 0x03);
    }
  }
  
  return lError;
}

